/*******
*
*  amfiocli.prg by Aleksander Czajczynski <hb/at/fki.pl> 2012
*
*  amfiocli.prg - Basic routines for communications using AMFIO
*                 over standard IO pipes and TCP/IP
*
********/

#include "hbamf.ch"
#include "hbsocket.ch"
#include "hbclass.ch"

// internal Harbour functions used in this file
#xtranslate MethodName() => __GetMessage()

STATIC s_nLastRefID := 0
STATIC s_aRefIDs[ AMFIO_MAXREFID ]
STATIC s_hConnections
THREAD STATIC s_pConnectionSocket
THREAD STATIC s_hLPP
THREAD STATIC s_symConvIn
THREAD STATIC s_symConvOut
THREAD STATIC s_nTimeout
STATIC s_lExit := .F.

// for Object as Service Extensions
STATIC s_nLastOid := 0 /* TOFIX(?)/QUESTION: is mutex needed to synchronize this */
STATIC s_pLastConnection

THREAD STATIC s_pConvInConnection

#define _AMFIOCLI_COUNTER 1
#define _AMFIOCLI_SOCKET 2
#define _AMFIOCLI_LPP 3
#define _AMFIOCLI_SYMCONVIN 4
#define _AMFIOCLI_SYMCONVOUT 5
#define _AMFIOCLI_TIMEOUT 6

FUNCTION AMFIO_Connect( cServer, nPort, nTimeout, symConvIn, symConvOut, lObjectExt /* , symOnConnect */ )
   LOCAL aConn, pConnectionSocket, cHost

   cServer := Lower( AllTrim( cServer ) )

   IF !Empty( lObjectExt )
      IF HB_IsLogical( symConvIn ) .AND. symConvIn
         symConvIn := @AMFIO_ClientConvIn()
      ENDIF
      IF HB_IsLogical( symConvOut ) .AND. symConvOut
         symConvOut := @AMFIO_ClientConvOut()
      ENDIF
   ENDIF

   IF Empty( s_hConnections )
      s_hConnections := { => }
   ELSEIF HB_HHasKey( s_hConnections, cServer + ":" + HB_NtoS( nPort ) )
      aConn := s_hConnections[ cServer + ":" + HB_NtoS( nPort ) ]
      s_pConnectionSocket := aConn[ _AMFIOCLI_SOCKET ]
      s_hLPP := aConn[ _AMFIOCLI_LPP ]
      s_symConvIn := aConn[ _AMFIOCLI_SYMCONVIN ]
      s_symConvOut := aConn[ _AMFIOCLI_SYMCONVOUT ]
      s_nTimeout := aConn[ _AMFIOCLI_TIMEOUT ]
      aConn[ _AMFIOCLI_COUNTER ] ++
      RETURN .T.
   ENDIF

   cHost := HB_SocketResolveAddr( cServer )
   IF Empty( cHost )
      RETURN .F.
   ENDIF

   pConnectionSocket := HB_SocketOpen()
   IF HB_SocketConnect( pConnectionSocket, { HB_SOCKET_AF_INET, cHost, nPort }, nTimeout )
      HB_SocketSend( pConnectionSocket, HB_BChar( 12 ), , , nTimeout )
      aConn := Array( _AMFIOCLI_TIMEOUT )
      aConn[ _AMFIOCLI_SOCKET ] := s_pConnectionSocket := pConnectionSocket
      aConn[ _AMFIOCLI_LPP ] := s_hLPP := HB_lppCreate( pConnectionSocket )
      aConn[ _AMFIOCLI_SYMCONVIN ] := s_symConvIn := symConvIn
      aConn[ _AMFIOCLI_SYMCONVOUT ] := s_symConvOut := symConvOut
      aConn[ _AMFIOCLI_TIMEOUT ] := s_nTimeout := nTimeout
      aConn[ _AMFIOCLI_COUNTER ] := 1
      s_hConnections[ cServer + ":" + HB_NtoS( nPort ) ] := aConn
      s_hConnections[ pConnectionSocket ] := aConn
      RETURN .T.
   ENDIF
   HB_SocketClose( pConnectionSocket )

   RETURN .F.

FUNCTION AMFIO_Disconnect( cServer, nPort )

   LOCAL aConn

   cServer := Lower( AllTrim( cServer ) )

   IF HB_HHasKey( s_hConnections, cServer + ":" + HB_NtoS( nPort ) )
      aConn := cServer + ":" + HB_NtoS( nPort )
      IF aConn[ _AMFIOCLI_COUNTER ] == 1
         HB_LppDestroy( aConn[ _AMFIOCLI_LPP ] )
         HB_SocketShutdown( aConn[ _AMFIOCLI_SOCKET ] )
         HB_SocketClose( aConn[ _AMFIOCLI_SOCKET ] )
         HB_HDel( s_hConnections, cServer + ":" + HB_NtoS( nPort ) )
         HB_HDel( s_hConnections, aConn[ _AMFIOCLI_SOCKET ] )

         IF aConn[ _AMFIOCLI_SOCKET ] == s_pConnectionSocket
            s_pConnectionSocket := NIL
            s_hLPP := NIL
            s_symConvIn := NIL
            s_symConvOut := NIL
            s_nTimeout := NIL
         ENDIF
      ELSE
         aConn[ _AMFIOCLI_COUNTER ] --
      ENDIF
      RETURN .T.
   ENDIF

   RETURN .F.

FUNCTION AMFIO_Request( pConnectionSocket, nType, xResponder, ... )

   LOCAL cPacket
/*   LOCAL nLen, nChecksum, i */
   LOCAL a, e
   LOCAL cRefID
   
   IF Empty( pConnectionSocket )
      e := ErrorNew()
      e:SubSystem := "AMFIO"
      e:Operation := "REQUEST"
      e:Description := "AMFIO not connected"
      e:Severity := 2
      Eval( ErrorBlock(), e )
      Break( e )
   ENDIF

   IF nType # AMFIO_PROCEXEC
      cRefID := LEW2Bin( GenRefId( xResponder ) )
   ELSE
      cRefID := ""
   ENDIF

   a := HB_AParams()
   ADel( a, 1 )
   ADel( a, 1 )
   ADel( a, 1 )
   ASize( a, Len( a ) - 3 )

   cPacket := HB_BChar( nType ) + cRefID + AMF3_Encode( a, s_symConvOut )
/*
   nCheckSum := 0
   // TODO: C-level XOR
   nLen := HB_BLen( cPacket )
   FOR i := 1 TO nLen
      nCheckSum := HB_BitXOR( nCheckSum, HB_BCode( HB_BSubStr( cPacket, i, 1 ) ) )
   NEXT
*/
   cPacket := LEU2Bin( HB_BLen( cPacket ) + 1 ) + cPacket + XORCount( cPacket )

   IF nType == AMFIO_PROCEXEC .OR. !Empty( xResponder )
      IF HB_SocketSend( pConnectionSocket, cPacket ) > - 1
         RETURN .T.
      ELSE
         e := WriteErrorNew()
         Eval( ErrorBlock(), e )
         Break( e )
         //         RETURN .F.
      ENDIF
   ENDIF

   IF HB_SocketSend( pConnectionSocket, cPacket, , , s_nTimeout ) > - 1
      RETURN AMFIO_ClientRecvLoop( cRefID, s_nTimeout, pConnectionSocket )
   ELSE
      e := WriteErrorNew()
      Eval( ErrorBlock(), e )
      Break( e )
   ENDIF

   RETURN NIL

STATIC FUNCTION WriteErrorNew()

   LOCAL e := ErrorNew()

   e:SubSystem := "AMFIO"
   e:Operation := "REQUEST"
   e:Description := "write error"
   e:Severity := 2

   RETURN e

FUNCTION AMFIO_ClientRecvLoop( cReturnRefID, nTimeout, pConnectionSocket )

   LOCAL xRet
   LOCAL cBuf := Space( 4096 )
   LOCAL cLastRefID
   LOCAL nLPPTimeout
   LOCAL hLPP
   LOCAL symConvIn

   IF Empty( pConnectionSocket )
      pConnectionSocket := s_pConnectionSocket
      hLPP := s_hLPP
      symConvIn := s_symConvIn
   ELSEIF pConnectionSocket == s_pConnectionSocket
      hLPP := s_hLPP
      symConvIn := s_symConvIn
   ELSEIF HB_HHasKey( s_hConnections, pConnectionSocket )
      hLPP := s_hConnections[ pConnectionSocket ][ _AMFIOCLI_LPP ]
      symConvIn := s_hConnections[ pConnectionSocket ][ _AMFIOCLI_SYMCONVIN ]
   ELSE
      RETURN NIL
   ENDIF

   s_pConvInConnection := pConnectionSocket

   // when nTimeout < 0, function exits after single iteration
   IF HB_IsNumeric( nTimeout )
      IF nTimeout <= 0
         nLPPTimeout := 0
      ELSE
         nLPPTimeout := 1000
      ENDIF
   ENDIF

   DO WHILE .T.

      IF ! HB_LppRecv( hLPP, @cBuf, nLPPTimeout )
         IF HB_SocketGetError( pConnectionSocket ) == HB_SOCKET_ERR_TIMEOUT
            IF s_lExit
               EXIT
            ELSEIF HB_IsNumeric( nTimeout )
               nTimeout -= nLPPTimeout
               IF nTimeout <= 0
                  EXIT
               ELSE
                  LOOP
               ENDIF
            ELSE
               LOOP
            ENDIF
         ELSE
            EXIT
         ENDIF
      ENDIF

      IF HB_IsString( cReturnRefID )
         xRet := AMFIO_HandleResponse( cBuf, symConvIn, @cLastRefID )
         IF HB_IsString( cLastRefId ) .AND. cLastRefID == cReturnRefID
            RETURN xRet
         ENDIF
      ELSE
         AMFIO_HandleResponse( cBuf, symConvIn )
      ENDIF

      IF s_lExit
         RETURN NIL
      ENDIF

   ENDDO

   RETURN NIL

STATIC FUNCTION AMFIO_HandleResponse( cBuf, symConvIn, cReturnRefID )

   LOCAL nLen, nRefID, xResponder, xRespDecision, xRet

   nLen := HB_BLen( cBuf )
     
   IF ! XORCHECK( HB_BLeft( cBuf, nLen - 1 ), HB_BRight( cBuf, 1 ) )
      s_lExit := .T.
      RETURN NIL
   ENDIF

   cReturnRefID := HB_BLeft( cBuf, 2 )
   IF !( cReturnRefID == HB_BChar( 0 ) + HB_BChar( 0 ) )
      nRefID := Bin2W( cReturnRefID )
      xRet := AMF3_Decode( HB_BSubStr( cBuf, 3 ), symConvIn )
      xResponder := s_aRefIDs[ nRefID ]
      IF HB_IsSymbol( xResponder )
         xRespDecision := xResponder:Exec( xRet )
      ELSEIF HB_IsBlock( xResponder )
         xRespDecision := Eval( xResponder, xRet )
      ENDIF
      IF HB_IsLogical( xRespDecision )
         IF xRespDecision
            s_aRefIDs[ nRefID ] := NIL
            // if responder returns a false value, then it's waiting for subsequent messages under this refID,
            // so we're not releasing this particular refID
         ENDIF
      ELSE
         s_aRefIDs[ nRefID ] := NIL
      ENDIF
      RETURN xRet
      // ELSE - future reserved, zero refid exists if server makes requests to client
   ENDIF

   RETURN NIL

STATIC FUNCTION GenRefId( xResponder )

   LOCAL nSkipped := 0, e

   IF ++ s_nLastRefID > AMFIO_MAXREFID
      s_nlastRefID := 1
   ENDIF

   DO WHILE !Empty( s_arefIDs[ s_nLastRefID ] )

      IF s_nlastRefID > AMFIO_MAXREFID
         s_nlastRefID := 1
      ELSE
         s_nlastRefID ++
      ENDIF

      IF ++ nSkipped == AMFIO_MAXREFID
         e := ErrorNew()
         e:SubSystem := "AMFIO"
         e:Operation := "GENREFID"
         e:Description := "request reference ID's exhausted"
         e:Severity := 2
         Eval( ErrorBlock(), e )
         Break( e )
      ENDIF

   ENDDO

   IF !Empty( xResponder )
      s_arefIDs[ s_nLastRefID ] := xResponder
   ELSE
      s_arefIDs[ s_nLastRefID ] := .T.
   ENDIF

   RETURN s_nLastRefID

FUNCTION AMFIO_ProcExists( cProcName )

   RETURN AMFIO_Request( s_pConnectionSocket, AMFIO_PROCEXISTS, , cProcName )

FUNCTION AMFIO_ProcExec( cProcName, ... )

   RETURN AMFIO_Request( s_pConnectionSocket, AMFIO_PROCEXEC, , cProcName, ... )

FUNCTION AMFIO_FuncExec( cFuncName, ... )

   RETURN AMFIO_Request( s_pConnectionSocket, AMFIO_FUNCEXEC, , cFuncName, ... )

FUNCTION AMFIO_FuncExecR( cFuncName, xResponder, ... )

   RETURN AMFIO_Request( s_pConnectionSocket, AMFIO_FUNCEXEC, xResponder, cFuncName, ... )

FUNCTION AMFIO_ClientConvIn( xIn )

   IF ValType( xIn ) = "O"
      IF xIn:className == "OBJAMFPROXY"
         xIn := ObjRpcAMF():New( xIn:getID() )
         xIn:RpcSetConnection( s_pConvInConnection )
      ENDIF
   ENDIF

   RETURN xIn

FUNCTION AMFIO_ClientConvOut( xOut )

   IF ValType( xOut ) = "O"
      IF xOut:className == "OBJRPCAMF"
         // TODO: check if object belongs to this connection
         xOut := ObjAMFProxy():New( xOut:RpcOid )
      ENDIF
   ENDIF

   RETURN xOut

CLASS ObjRpcAMF

   METHOD New( nOid, cClass, hCached ) Constructor
   DESTRUCTOR RpcDestroy

   ERROR HANDLER noMessage
   METHOD msgNotFound

   PROTECTED:
   VAR nVersion INIT 0
   VAR nRpcOid
   VAR pConnection
   VAR cRealClass
   VAR hCachedData

   EXPORTED:
   METHOD RpcCacheClear
   METHOD RpcSetConnection
   ACCESS RealClass INLINE ::cRealClass
   ACCESS RpcOid INLINE ::nRpcOid

END CLASS

METHOD New( nOid, cClass, hCached )

   ::nRpcOid := nOid
   ::cRealClass := cClass
   ::hCachedData := hCached

   RETURN self

METHOD noMessage( ... ) CLASS ObjRpcAMF

   RETURN ::msgNotFound( MethodName(), ... )

METHOD msgNotFound( cMessage, ... ) CLASS ObjRpcAMF

   IF PCount() = 1 .AND. HB_BLeft( cMessage, 1 ) # "_"
      IF !Empty( ::hCachedData ) .AND. HB_HHasKey( ::hCachedData, cMessage )
         RETURN ::hCachedData[ cMessage ]
      ENDIF
   ENDIF

   IF !( !Empty( s_pLastConnection ) .AND. s_pLastConnection == ::pConnection .AND. s_nLastOid == ::nRpcOid )
      IF AMFIO_ObjectWith( ::pConnection, ::nRpcOid ) # ::nRpcOid // ObjectWith
         RETURN NIL // TODO(?): Throw Error
      ELSE
         s_pLastConnection := ::pConnection
         s_nLastOid := ::nRpcOid
      ENDIF
   ELSE
      s_pLastConnection := ::pConnection
      s_nLastOid := ::nRpcOid
   ENDIF
   
   RETURN AMFIO_ObjectSend( ::pConnection, cMessage, ... )

METHOD RpcSetConnection( pConnection ) CLASS ObjRpcAMF

   IF Empty( ::pConnection )
      IF Empty( pConnection )
         ::pConnection := s_pConnectionSocket
      ELSE
         ::pConnection := pConnection
      ENDIF
   ENDIF

   RETURN self

METHOD RpcCacheClear() CLASS ObjRpcAMF

   ::hCachedData := NIL

   RETURN self

METHOD RpcDestroy() CLASS ObjRpcAMF

   IF !Empty( ::pConnection )
      AMFIO_ObjectDel( ::pConnection, ::nRpcOid )
   ENDIF

   RETURN self

/*
FUNCTION NETIO_ObjExec( pConnection, cFunc, ... )
   LOCAL xRet := NETIO_FuncExec( pConnection, cFunc, ... )

   IF ValType( xRet ) = "O"
      IF xRet:className == "ERROR"
         Eval( ErrorBlock(), xRet )
         Break( xRet )
         // RETURN NIL
      ELSEIF __objHasMsg( xRet, "ObjRpcAMF" )
         xRet:RpcSetConnection()
      ENDIF
   ENDIF

   RETURN xRet
*/

FUNCTION AMFIO_ObjectNew( cClassName, ... )

   LOCAL nId := AMFIO_Request( s_pConnectionSocket, AMFIO_OBJECTNEW, , cClassName, ... )
   LOCAL o

   IF HB_IsNumeric( nId ) .AND. nId > 0
      o := ObjRpcAMF():New( nId )
      o:RpcSetConnection()
      RETURN o
   ENDIF
 
   RETURN NIL

FUNCTION AMFIO_ObjectDel( pConnection, nObject )

   RETURN AMFIO_Request( pConnection, AMFIO_OBJECTDEL, , nObject )

FUNCTION AMFIO_ObjectWith( pConnection, nObject )

   RETURN AMFIO_Request( pConnection, AMFIO_OBJECTWITH, , nObject )

FUNCTION AMFIO_ObjectSend( pConnection, cMethodName, ... )

   RETURN AMFIO_Request( pConnection, AMFIO_OBJECTSEND, , cMethodName, ... )

FUNCTION AMFIO_ObjectGetV( pConnection, aVarsToGet )

   RETURN AMFIO_Request( pConnection, AMFIO_OBJECTGETV, , aVarsToGet )
