/*******
*
*  amfiosrv.prg by
*  Ilina Stoilkovska <anili100/at/gmail.com> 2011
*  Aleksander Czajczynski <hb/at/fki.pl> 2011-2012
*
*  amfiosrv.prg - Basic routines for communications using AMFIO
*                 over standard IO pipes and TCP/IP
*
********/

#include "hbamf.ch"
#include "hbclass.ch"
#include "hbsocket.ch"
#include "fileio.ch"
#include "common.ch"
#include "hboo.ch"
#include "hbthread.ch"

// internal Harbour functions used in this file
#xtranslate String2Symbol( <exp> ) => __DynsN2Sym( <exp> )
#xtranslate ObjectHasMessage( <object>, <message> ) => __objHasMsg( <object>, <message> )
#xtranslate ObjectSend( <exp,...> ) => __objSendMsg( <exp> )
#xtranslate MethodName() => __GetMessage()

THREAD STATIC s_functions
THREAD STATIC s_objects
THREAD STATIC s_selectedObject
THREAD STATIC s_position
THREAD STATIC s_cRefID
THREAD STATIC s_bSendBlock
STATIC s_lExit := .F.

// this is in fact needed by lower .c level code, i couldn't figure out how to request from .c
REQUEST __CLSMSGTYPE

FUNCTION AMFIO_MTSERVER( nPort, cIfAddr, hAnonFuncs, lUserControl, symConvIn, symConvOut, lObjectExt )

   LOCAL pListenSocket

   IF !Empty( lObjectExt )
      IF HB_IsLogical( symConvIn ) .AND. symConvIn
         symConvIn := @AMFIO_ObjConvIn()
      ENDIF
      IF HB_IsLogical( symConvOut ) .AND. symConvOut
         symConvOut := @AMFIO_ObjConvOut()
      ENDIF
   ELSE
      IF !HB_IsSymbol( symConvIn )
         symConvIn := NIL
      ENDIF
      IF !HB_IsSymbol( symConvOut )
         symConvOut := NIL
      ENDIF
   ENDIF

   IF HB_MTVM()
      
      s_lExit := .F.
      
      pListenSocket := HB_SocketOpen()
      IF ! HB_SocketBind( pListenSocket, { HB_SOCKET_AF_INET, cIfAddr, nPort } )
         ? "bind() error", HB_socketGetError()
         RETURN - 1
      ELSEIF ! HB_SocketListen( pListenSocket )
         ? "listen() error", HB_socketGetError()
         RETURN - 1
      ENDIF

      HB_ThreadDetach( HB_ThreadStart( HB_THREAD_INHERIT_PUBLIC, @AMFIO_SRVLOOP(), pListenSocket, @AMFIO_SERVER(), hAnonFuncs, lUserControl, symConvIn, symConvOut, lObjectExt ) )

   ENDIF

   RETURN pListenSocket

FUNCTION AMFIO_SRVLOOP( pListenSocket, symServer, ... )

   LOCAL pConnectionSocket
   LOCAL aRemote := {}

   DO WHILE .T.

      pConnectionSocket := HB_SocketAccept( pListenSocket, @aRemote, 50 )
      
      IF ! Empty( pConnectionSocket )
         IF s_lExit == .T.
            HB_SocketShutdown( pConnectionSocket )
            HB_SocketClose( pConnectionSocket )
            RETURN 0
         ENDIF
         HB_ThreadDetach( HB_ThreadStart( HB_THREAD_INHERIT_PUBLIC, symServer, pConnectionSocket, ... ) )
      ELSEIF s_lExit == .T.
         RETURN 0
      ENDIF
   ENDDO

   RETURN 0
   
PROCEDURE AMFIO_INIT( hAnonFuncs, lUserControl, symFuncRedir )

   LOCAL i, c

   s_functions := { => }
   s_objects := { => }
   s_position := 1
   HB_HSetCaseMatch( s_functions, .F. )

   IF !Empty( lUserControl )
      s_functions[ "AUTH" ] := String2Symbol( "AUTH" ) // @Auth() would cause link time error for apps without User Access Control
   ENDIF
   
   IF !Empty( hAnonFuncs )
      IF Empty( symFuncRedir ) // do not optimize if we have redirection to a function
         IF Empty( HB_HValueAt( hAnonFuncs, 1 ) ) // get function symbols into the hash
            FOR i := 1 TO Len( hAnonFuncs )       // delete not existent functions
               IF HB_IsFunction( c := HB_HKeyAt( hAnonFuncs, i ) )
                  hAnonFuncs[ c ] := String2Symbol( c )
               ELSE
                  HB_HDelAt( hAnonFuncs, i -- )
               ENDIF
            NEXT
         ENDIF
      ENDIF

      IF Len( hAnonFuncs ) > 0
         s_functions := HB_HMerge( s_functions, hAnonFuncs )
      ENDIF

   ENDIF

   RETURN

FUNCTION AMFIO_GETALLOWEDCALLS()

   RETURN s_functions

PROCEDURE AMFIO_SETALLOWEDCALLS( hNewHashList )

   s_functions := hNewHashList

   RETURN

PROCEDURE AMFIO_SERVER( pConnectionSocket, hAnonFuncs, lUserControl, symConvIn, symConvOut, lObjectExt, symFuncRedir )

   LOCAL cBuf := Space( 4096 )
   LOCAL hLPP, cPacket

   AMFIO_INIT( hAnonFuncs, lUserControl, symFuncRedir )
   
   HB_SocketRecv( pConnectionSocket, @cBuf, 1 )
   IF cBuf = '<'
      HB_SocketRecv( pConnectionSocket, @cBuf, HB_BLen( "policy-file-request/>" ) )
      IF cBuf = "policy-file-request/>"
         sendPolicy( pConnectionSocket )
         RETURN
      ENDIF
   ENDIF
      
   IF cBuf # HB_BChar( 12 )
      HB_SocketShutdown( pConnectionSocket )
      HB_SocketClose( pConnectionSocket )
      RETURN
   ENDIF

   hLPP := HB_LppCreate( pConnectionSocket )
   s_bSendBlock := { |c| HB_SocketSend( pConnectionSocket, LEU2Bin( HB_BLen( c ) ) + c ) }

   DO WHILE .T.
      
      IF ! HB_LppRecv( hLPP, @cBuf, 1000 )
         IF HB_SocketGetError( pConnectionSocket ) == HB_SOCKET_ERR_TIMEOUT
            IF s_lExit
               EXIT
            ELSE
               LOOP
            ENDIF
         ELSE
            EXIT
         ENDIF
      ENDIF

      cPacket := AMFIO_HandleRequest( @cBuf, symConvIn, symConvOut, lObjectExt, symFuncRedir )

      IF cPacket != NIL
         cPacket := LEU2Bin( HB_BLen( cPacket ) ) + cPacket
         HB_SocketSend( pConnectionSocket, cPacket )
      ENDIF

      IF s_lExit
         EXIT
      ENDIF
      
   ENDDO

   s_bSendBlock := NIL
   HB_LppDestroy( hLPP )
   HB_SocketShutdown( pConnectionSocket )
   HB_SocketClose( pConnectionSocket )

   RETURN
   
STATIC FUNCTION SendPolicy( hSocket )

   STATIC cContent
   
   IF Empty( cContent )
      cContent := ReadStream( "policy.xml" ) + HB_BChar( 0 )
   ENDIF
   
   HB_SocketSend( hSocket, cContent, HB_BLen( cContent ) )
   HB_Idlesleep( 5 )
   HB_SocketShutdown( hSocket )
   HB_SocketClose( hSocket )

   RETURN 0
   
STATIC FUNCTION ReadStream( cFile )

   LOCAL cBuffer     := Space( BLOCK_SIZE )
   LOCAL nFileHandle := FOpen( cFile )
   LOCAL cOut     := ""
   LOCAL nRead

   IF FError() <> 0
      ? "Error opening file:", FError()
      QUIT
   ENDIF

   DO WHILE .T.
      nRead := FRead( nFileHandle, @cBuffer, BLOCK_SIZE )
      IF nRead == BLOCK_SIZE
         cOut += cBuffer
      ELSE
         cOut += HB_BLeft( cBuffer, nRead )
         EXIT
      ENDIF
   ENDDO

   FClose( nFileHandle )

   RETURN cOut

FUNCTION AMFIO_HandleRequest( cBuf, symConvIn, symConvOut, lObjectExt, symFuncRedir )

   LOCAL nLen, cRefID, pItem, cType, xVal, aParams, i, symFunc

   nLen := HB_BLen( cBuf )
      
   IF ! XORCheck( HB_BLeft( cBuf, nLen - 1 ), HB_BRight( cBuf, 1 ) )
      RETURN NIL
   ENDIF
      
   cBuf := HB_BLeft( cBuf, -- nLen )
   cType := HB_BLeft( cBuf, 1 )
   cBuf := HB_BRight( cBuf, -- nLen )

   IF HB_BCode( cType ) != AMFIO_PROCEXEC
      cRefID := HB_BLeft( cBuf, 2 )
      cBuf := HB_BRight( cBuf, nLen - 2 )
      //         nLen := nLen - 2
   ELSE
      cRefID := NIL
   ENDIF
      
   xVal := AMF3_Decode( cBuf, symConvIn )

   IF cRefID != HB_BChar( 0 ) + HB_BChar( 0 ) .AND. ValType( xVal ) = "A" .AND. Len( xVal ) > 0

      s_cRefID := cRefID

      SWITCH HB_BCode( cType )

      CASE AMFIO_PROCEXISTS

         IF AMFIO_FuncIsAllowed( xVal[ 1 ] )
            pItem := .T.
         ELSE
            pItem := .F.
         ENDIF

         EXIT

      CASE AMFIO_PROCEXEC

         IF AMFIO_FuncIsAllowed( xVal[ 1 ], @symFunc )
            // symFunc := String2Symbol( xVal[ 1 ] )
            IF HB_IsSymbol( symFuncRedir )
               symFuncRedir:Exec( HB_ArrayToParams( xVal ) )
            ELSE
               aParams := HB_ADel( xVal, 1, .T. )
               symFunc:Exec( HB_ArrayToParams( aParams ) )
            ENDIF
            pItem := .T.
         ELSE
            pItem := .F.
         ENDIF

         EXIT
        
      CASE AMFIO_FUNCEXEC

         IF AMFIO_FuncIsAllowed( xVal[ 1 ], @symFunc )
            IF HB_IsSymbol( symFuncRedir )
               pItem := symFuncRedir:Exec( HB_ArrayToParams( xVal ) )
            ELSE
               aParams := HB_ADel( xVal, 1, .T. )
               pItem := symFunc:Exec( HB_ArrayToParams( aParams ) )
            ENDIF
         ENDIF

         EXIT
        
         OTHERWISE

         IF !Empty( lObjectExt )

            SWITCH HB_BCode( ctype )
            CASE AMFIO_OBJECTNEW

               IF AMFIO_FuncIsAllowed( xVal[ 1 ], @symFunc )
                  // symFunc := String2Symbol( xVal[ 1 ] )
                  IF HB_IsSymbol( symFuncRedir )
                     s_objects[ s_position ] := symFuncRedir:Exec( xVal[ 1 ] ):New( HB_ArrayToParams( HB_ADel( xVal, 1, .T. ) ) )
                  ELSE
                     aParams := HB_ADel( xVal, 1, .T. )
                     s_objects[ s_position ] := symFunc:Exec():New( HB_ArrayToParams( aParams ) )
                  ENDIF
                  pItem := s_position
                  s_position ++
               ENDIF

               EXIT
      
            CASE AMFIO_OBJECTSEND

               IF ValType( s_selectedObject ) = "O"
                  IF ObjectHasMessage( s_selectedObject, xVal[ 1 ] )
                     aParams := HB_AIns( xVal, 1, s_selectedObject, .T. )
                     pItem := ObjectSend( HB_ArrayToParams( aParams ) )
                  ENDIF
               ENDIF

               EXIT

            CASE AMFIO_OBJECTGETV

               IF ValType( s_selectedObject ) = "O"
                  aParams := NIL
                  IF Empty( xVal[ 1 ] )

                     IF ObjectHasMessage( s_selectedObject, "RPCHINTCACHE" )
                        IF ValType( xVal[ 1 ] ) = "N" // Empty() i "N" == 0, klient prosi o nazwy zmiennych i wartoci
                           aParams := s_selectedObject:RpcHintCache
                           xVal := 1
                        ELSE
                           aParams := s_selectedObject:RpcHintCache
                           xVal := 0
                        ENDIF
                     ENDIF
                  ELSEIF ValType( xVal[ 1 ] ) = "A"
                     aParams := xVal[ 1 ]
                     xVal := 0
                  ENDIF

                  IF ValType( aParams ) = "A"
                     pItem := Array( Len( aParams ) + xVal )
                     FOR i := 1 TO Len( aParams )
                        IF ValType( aParams[ i ] ) = "C"
                           IF ObjectHasMessage( s_selectedObject, aParams[ i ] )
                              pItem[ i + xVal ] := s_selectedObject:&( aParams[ i ] )
                           ENDIF
                        ENDIF
                     NEXT
                     IF xVal > 0
                        pItem[ 1 ] := s_selectedObject:RpcHintCache
                     ENDIF
                  ENDIF

               ENDIF

               EXIT

            CASE AMFIO_OBJECTWITH

               IF HB_HHasKey( s_objects, xVal[ 1 ] )
                  s_selectedObject := s_objects[ xVal[ 1 ] ]
                  pItem := xVal[ 1 ]
               ELSE
                  s_selectedObject := NIL
               ENDIF

               EXIT

            CASE AMFIO_OBJECTDEL

               IF HB_HHasKey( s_objects, xVal[ 1 ] )
                  IF s_selectedObject == s_objects[ xVal[ 1 ] ]
                     s_selectedObject := NIL
                  ENDIF
                  s_objects[ xVal[ 1 ] ] := NIL
                  pItem := xVal[ 1 ]
               ELSE
                  pItem := - 1
               ENDIF

            END SWITCH

         ENDIF

      END SWITCH

      s_cRefID := NIL

   ELSE

      RETURN NIL

   ENDIF

   RETURN AMFIO_Response( cRefID, pItem, symConvOut )

FUNCTION AMFIO_Response( cRefID, pItem, symConvOut )

   IF ValType( pItem ) = "O" .AND. pItem:className == "RAWAMF"
      pItem := pItem:GetData()
   ELSE
      pItem := AMF3_Encode( pItem, symConvOut )
   ENDIF

   RETURN EncodePacket( cRefID, pItem )

FUNCTION AMFIO_CurrentRefId()

   RETURN s_cRefID

PROCEDURE AMFIO_ThreadSendBlock( bBlock )

   s_bSendBlock := bBlock

   RETURN

FUNCTION AMFIO_CyclicResponse( cRefID, pItem, symConvOut )

   IF ValType( s_bSendBlock ) = "B"
      pItem := AMFIO_Response( cRefID, pItem, symConvOut )
      RETURN Eval( s_bSendBlock, pItem )
   ENDIF

   RETURN NIL

STATIC FUNCTION AMFIO_FuncIsAllowed( cFunc, symFunc )

   // TODO: with empty allowed functions (s_Functions) hash, IsFunction will fail when symFuncRedir is active in AMFIO_Server

   IF !HB_IsHash( s_Functions )
      IF HB_IsFunction( cFunc )
         symFunc := String2Symbol( cFunc )
         RETURN .T.
      ELSE
         RETURN .F.
      ENDIF
   ELSEIF HB_hHasKey( s_functions, cFunc )
      symFunc := s_functions[ cFunc ]
      RETURN .T.
   ENDIF

   RETURN .F.
   
STATIC FUNCTION EncodePacket( cRefID, cBuf )

   LOCAL cPacket
   
   IF cRefID != NIL
      cPacket := cRefID + cBuf
   ELSE
      cPacket := HB_BChar( 0 ) + HB_BChar( 0 ) + cBuf   // ?
   ENDIF
   
   cPacket := cPacket + XORCount( cPacket )
   
   RETURN cPacket
   
FUNCTION AMFIO_SHUTDOWN()

   RETURN s_lExit := .T.

FUNCTION AMFIO_ObjConvIn( xIn )

   IF ValType( xIn ) = "O"
      IF ObjectHasMessage( xIn, "OBJAMFPROXY" )
         BEGIN SEQUENCE WITH { |e| Break( e ) }
            xIn := s_objects[ xIn:getID() ]
         RECOVER
            xIn := NIL
         END SEQUENCE
      ENDIF
   ENDIF

   RETURN xIn

FUNCTION AMFIO_ObjConvOut( xOut )

   IF ValType( xOut ) = "O"
      IF xOut == s_selectedObject // if it equals selected object
         RETURN NIL               // you probably run a method with RETURN self
         // i don't think we need it to again and again
         // on wire
      ELSEIF !ObjectHasMessage( xOut, "RPCSERIALIZE" )
         s_objects[ s_position ] := xOut
         xOut := ObjAMFProxy():New( s_position )

         s_position ++
      ENDIF
   ENDIF

   RETURN xOut

   INIT PROCEDURE AMFClasses
   ObjAMF()
   ObjAMFProxy()

   RETURN
   
CLASS ObjAMFProxy

   METHOD New( nId ) Constructor
   METHOD readExternal( cIn )
   METHOD writeExternal()
   METHOD getID()
   CLASSDATA EXTERNALIZABLE INIT .T.
   
   PROTECTED:
   VAR nID
   
END CLASS

METHOD New( nId ) CLASS ObjAMFProxy

   ::nId := nId

   RETURN self

METHOD readExternal( cIn ) CLASS ObjAMFProxy

   ::nID := LEBin2U( cIn )

   RETURN 4
   
METHOD writeExternal() CLASS ObjAMFProxy

   RETURN LEU2Bin( ::nID )
   
METHOD getID() CLASS ObjAMFProxy

   RETURN ::nID

CLASS RawAMF

   METHOD New( cData ) Constructor
   METHOD GetData INLINE ::cData
   
   PROTECTED:
   VAR cData
    
END CLASS

METHOD New( cData ) CLASS RawAMF

   ::cData := cData

   RETURN self
      
CLASS ObjAMF

   METHOD New( hCachedData ) Constructor
   ERROR HANDLER noMessage
   METHOD msgNotFound

   PROTECTED:
   VAR nVersion INIT 0
   VAR cRealClass
   VAR hCachedData
   VAR nRpcOid
   VAR pConnection
   VAR nID
   
   EXPORTED:
   ACCESS RealClass INLINE ::cRealClass
   ACCESS RpcOid INLINE ::nRpcOid

END CLASS

METHOD New( hCachedData ) CLASS ObjAMF

   ::hCachedData := hCachedData

   RETURN self

METHOD noMessage( ... ) CLASS ObjAMF

   RETURN ::msgNotFound( MethodName(), ... )
   
METHOD msgNotFound( cMessage, ... ) CLASS ObjAMF

   IF PCount() = 1 .AND. HB_BLeft( cMessage, 1 ) # "_"
      IF !Empty( ::hCachedData ) .AND. HB_HHasKey( ::hCachedData, cMessage )
         RETURN ::hCachedData[ cMessage ]
      ENDIF
   ELSEIF PCount() > 1 .AND. HB_BLeft( cMessage, 1 ) == "_"
      IF Empty( ::hCachedData )
         ::hCachedData := { => }
      ENDIF
      RETURN ::hCachedData[ HB_BSubStr( cMessage, 2 ) ] := HB_PValue( 2 )
   ENDIF

   RETURN NIL


