/*******
*
*  amfwssrv.prg by Aleksander Czajczynski <hb/at/fki.pl> 2012
*
*  amfwssrv.prg - Basic routines for communications using AMF
*                 over WebSocket
*
********/

#include "hbamf.ch"
#include "hbsocket.ch"
#include "common.ch"
#include "hboo.ch"
#include "hbthread.ch"

#define MAGIC_KEY "258EAFA5-E914-47DA-95CA-C5AB0DC85B11"
#define CRLF HB_BChar( 13 ) + HB_BChar( 10 )
#define MAX_MESSAGE_LEN 131070

#define FRAME_TEXT 0x01
#define FRAME_BINARY 0x02

STATIC s_lExit := .F.
THREAD STATIC s_pClientSocket

FUNCTION AMFWS_MTSERVER( nPort, cIfAddr, hAnonFuncs, lUserControl, symConvIn, symConvOut, lObjectExt )

   LOCAL pListenSocket

   IF !Empty( lObjectExt )
      IF HB_IsLogical( symConvIn ) .AND. symConvIn
         symConvIn := @AMFIO_ObjConvIn()
      ENDIF
      IF HB_IsLogical( symConvOut ) .AND. symConvOut
         symConvOut := @AMFIO_ObjConvOut()
      ENDIF
   ELSE
      IF !HB_IsSymbol( symConvIn )
         symConvIn := NIL
      ENDIF
      IF !HB_IsSymbol( symConvOut )
         symConvOut := NIL
      ENDIF
   ENDIF

   IF hb_mtvm()
      
      s_lExit := .F.
      
      pListenSocket := HB_SocketOpen()
      IF ! HB_SocketBind( pListenSocket, { HB_SOCKET_AF_INET, cIfAddr, nPort } )
         ? "bind() error", HB_SocketGetError()
         RETURN - 1
      ELSEIF ! HB_SocketListen( pListenSocket )
         ? "listen() error", HB_SocketGetError()
         RETURN - 1
      ENDIF

      hb_threadDetach( hb_threadStart( HB_THREAD_INHERIT_PUBLIC, @AMFWS_SRVLOOP(), pListenSocket, @AMFWS_SERVER(), hAnonFuncs, lUserControl, symConvIn, symConvOut, lObjectExt ) )

   ENDIF

   RETURN pListenSocket

FUNCTION AMFWS_SRVLOOP( pListenSocket, symServer, ... )

   LOCAL pConnectionSocket
   LOCAL aRemote := {}

   DO WHILE .T.

      pConnectionSocket := HB_SocketAccept( pListenSocket, @aRemote, 50 )
      
      IF ! Empty( pConnectionSocket )
         IF s_lExit == .T.
            HB_SocketShutdown( pConnectionSocket )
            HB_SocketClose( pConnectionSocket )
            RETURN 0
         ENDIF
         HB_ThreadDetach( hb_threadStart( HB_THREAD_INHERIT_PUBLIC, symServer, pConnectionSocket, ... ) )
      ELSEIF s_lExit == .T.
         RETURN 0
      ENDIF
   ENDDO

   RETURN 0

PROCEDURE AMFWS_SERVER( pConnectionSocket, hAnonFuncs, lUserControl, symConvIn, symConvOut, lObjectExt, symFuncRedir )

   LOCAL cBuf := Space( 4096 )
   LOCAL cPacket := Space( 4096 )
   LOCAL lHandShake := .F.
   LOCAL nLen, nPos
   LOCAL lFin, nOpCode, cPayLoad, lMask, cMask
   LOCAL nPayLoadLen
   LOCAL cResponse
   LOCAL lSocketError := .F. , oError

   cPacket := ""

   DO WHILE !lHandShake

      IF ( nLen := HB_SocketRecv( pConnectionSocket, @cBuf, 4096, 0, 1000 ) ) > 0
         IF HB_BLen( cPacket ) + nLen > 4096
            // HTTP request is going to be larger than 4096 bytes
            // we better disconnect such client
            lSocketError := .T.
            EXIT
         ELSE
            cPacket += HB_BLeft( cBuf, nLen )
            IF ( nPos := RAt( CRLF + CRLF, cPacket ) ) > 0
               IF AMFWS_HandShake( HB_BLeft( cPacket, nPos - 1 ), pConnectionSocket )
                  cPacket := HB_BSubStr( cPacket, nPos + 4 )
                  EXIT // handshake ok, advancing to stream loop
               ELSE
                  cPacket := HB_BSubStr( cPacket, nPos + 4 ) // we didn't recive a WebSocket request, wait for another one
               ENDIF
            ENDIF
         ENDIF
      ELSEIF HB_SocketGetError( pConnectionSocket ) # HB_SOCKET_ERR_TIMEOUT
         lSocketError := .T.
         EXIT
      ENDIF

      IF s_lExit
         EXIT
      ENDIF

   ENDDO

   IF s_lExit .OR. lSocketError
      HB_SocketShutdown( pConnectionSocket )
      HB_SocketClose( pConnectionSocket )
      RETURN
   ENDIF

   AMFIO_INIT( hAnonFuncs, lUserControl, symFuncRedir )
   s_pClientSocket := pConnectionSocket

   DO WHILE .T.

      IF !ReadUpto( 2, pConnectionSocket, @cBuf, @cPacket )
         EXIT
      ENDIF

      nOpCode := HB_BCode( HB_BLeft( cPacket, 1 ) )
      lFin := HB_BitTest( nOpCode, 7 )
      IF !lFin
         // message fragmenting not yet supported, we have to create
         // one another outer loop
         ? "fragmented messages not yet supported"
         EXIT
      ENDIF
      // RSV1, 6
      // RSV2, 5
      // RSV3, 4
      nOpCode := HB_BitAnd( nOpCode, 0x0F )

      nPayLoadLen := HB_BCode( HB_BSubStr( cPacket, 2, 1 ) )
      cPacket := HB_BSubStr( cPacket, 3 )

      lMask := HB_BitTest( nPayLoadLen, 7 )
      nPayLoadLen := HB_BitReset( nPayLoadLen, 7 )
      
      IF nPayLoadLen == 126 // 16-bit UINT
         IF !ReadUpto( 2, pConnectionSocket, @cBuf, @cPacket )
            EXIT
         ENDIF
         nPayLoadLen := BEBin2U( HB_BLeft( cPacket, 2 ) )
         cPacket := HB_BSubStr( cPacket, 3 )
      ELSEIF nPayLoadLen == 127 // 64-bit UINT
         IF !ReadUpto( 8, pConnectionSocket, @cBuf, @cPacket )
            EXIT
         ENDIF
         nPayLoadLen := BEBin2U( HB_BLeft( cPacket, 8 ) )
         cPacket := HB_BSubStr( cPacket, 9 )
      ENDIF

      IF nPayLoadLen > MAX_MESSAGE_LEN // message is too large for us, discard this client
         EXIT
      ENDIF

      IF lMask

         IF !ReadUpto( 4, pConnectionSocket, @cBuf, @cPacket )
            EXIT
         ENDIF

         cMask := HB_BLeft( cPacket, 4 )
         cPacket := HB_BSubStr( cPacket, 5 )

      ENDIF

      IF !ReadUpto( nPayLoadLen, pConnectionSocket, @cBuf, @cPacket )
         EXIT
      ENDIF

      cPayload := HB_BLeft( cPacket, nPayLoadLen )
      cPacket := HB_BSubStr( cPacket, nPayLoadLen + 1 )

      IF lMask
         cPayload := HB_StrXOR( cPayload, cMask )
      ENDIF

      IF nOpCode == FRAME_TEXT // we use FRAME_BINARY format now, but if AMF eventually goes through text WebSocket,
         // then it must not use (incorrect) UTF-8 sequences, it brokes connection in many ways
         cPayLoad := HB_Base64Decode( cPayload ) // one way to achieve this is Base64 encoding
      ENDIF

      ? "we received", cPayload
      cResponse := AMFIO_HandleRequest( @cPayLoad, symConvIn, symConvOut, lObjectExt, symFuncRedir )

      IF cResponse # NIL
         HB_SocketSend( pConnectionSocket, AMFWS_Response( cResponse, nOpCode ) )
      ENDIF

      IF s_lExit
         EXIT
      ENDIF

   ENDDO

   s_pClientSocket := NIL
   AMFIO_ThreadSendBlock( NIL )
   
   BEGIN SEQUENCE WITH { |e| Break( e ) }
      HB_SocketShutdown( pConnectionSocket )
      HB_SocketClose( pConnectionSocket )
   RECOVER USING oError
      /* this shouldn't ever happen */
      ? "unexpected error while shutting down socket " + oError:Description
   END SEQUENCE

   RETURN

STATIC FUNCTION ReadUpto( n, pConnectionSocket, cBuf, cPacket )
   LOCAL nLen, nRead := 0, oError, lRet := .T.
   BEGIN SEQUENCE WITH { |e| Break(e) }
      DO WHILE HB_BLen( cPacket ) < n .AND. !s_lExit
         IF ( nLen := HB_SocketRecv( pConnectionSocket, @cBuf, n - nRead, 0, 1000 ) ) > 0
            cPacket += HB_BLeft( cBuf, nLen )
            nRead += nLen
         ELSEIF HB_SocketGetError( pConnectionSocket ) # HB_SOCKET_ERR_TIMEOUT
            lRet := .F.
            EXIT
         ENDIF
      ENDDO
   RECOVER USING oError
      /* this shouldn't ever happen */
      ? "unexpected error on client socket " + oError:Description
      lRet := .F.
   END SEQUENCE
   IF s_lExit
      RETURN .F.
   ENDIF
   RETURN lRet

FUNCTION AMFWS_Response( cResponse, nOpCode )
   LOCAL nLen, cPayLoadLen

   IF !HB_IsNumeric( nOpCode )
      nOpCode := FRAME_BINARY // 0x01 is a text frame
   ENDIF

   IF nOpCode == FRAME_TEXT
      cResponse := HB_Base64Encode( cResponse )
   ENDIF

   nOpCode := HB_BitSet( nOpCode, 7 ) // set FIN frame marker

   nLen := HB_BLen( cResponse )
   IF nLen <= 125
      // HB_BitSet( nPayLoadLen, 7 ) // set this if you want string masking
      cPayLoadLen := HB_BChar( nLen )
   ELSEIF nLen >= 126 .and. nLen <= 65535
     // HB_BChar( HB_BitSet( 126, 7 ) ) // if string masking
     cPayLoadLen := HB_BChar( 126 ) // 16-bit UINT
     cPayLoadLen += BEW2Bin( nLen )
   ELSE
     // HB_BChar( HB_BitSet( 127, 7 ) ) // if string masking
     cPayLoadLen := HB_BChar( 127 ) // 64-bit UINT
     cPayLoadLen += BEU2Bin( nLen )
   ENDIF
   
   RETURN HB_BChar( nOpCode ) + cPayLoadLen + cResponse

// Based on a code shared by Antonio Linares and Daniel Garcia-Gil
STATIC FUNCTION AMFWS_HandShake( cBuffer, pConnectionSocket )
   LOCAL nLen, cContext, cKey, cSend
      
   cContext := AMFWS_GetContext( cBuffer, "Sec-WebSocket-Key" )
   IF !Empty( cContext )
      cKey := HB_Base64Encode( HB_SHA1( cContext + MAGIC_KEY, .T. ) ) // + "." add something to check that the handshake gets wrong
   
      cSend := "HTTP/1.1 101 Switching Protocols" + CRLF + ;
               "Upgrade: websocket" + CRLF + ;
               "Connection: Upgrade" + CRLF + ;
               "Sec-WebSocket-Accept: " + cKey + CRLF + CRLF

      nLen := HB_SocketSend( pConnectionSocket, cSend )

      IF nLen > 0
         RETURN .T.
      ENDIF
   ENDIF
   
   RETURN .F.

// Based on a code shared by Antonio Linares and Daniel Garcia-Gil
STATIC FUNCTION AMFWS_GetContext( cData, cContext )
  LOCAL cValue := ""
  LOCAL aLines
  LOCAL aSubLine
  LOCAL cRow
  
  aLines := hb_ATokens( cData, CRLF )
  FOR EACH cRow IN aLines
     IF cContext $ cRow
        aSubLine := hb_ATokens( cRow, ":" )
        cValue := AllTrim( aSubLine[ 2 ] )
        EXIT
     ENDIF
  NEXT

RETURN cValue

FUNCTION AMFWS_CyclicResponseBlock( symConvOut )
   LOCAL cRefId := AMFIO_CurrentRefId()
   LOCAL pSocket := s_pClientSocket
   RETURN { |uValue| HB_SocketSend( pSocket, AMFWS_Response( AMFIO_Response( cRefId, uValue, symConvOut ) ) ) }
   
FUNCTION AMFWS_SHUTDOWN()
   return s_lExit := .T.

